<?php
declare(strict_types=1);

require_once __DIR__ . '/../lib/auth.php';
require_once __DIR__ . '/../lib/db.php';
require_once __DIR__ . '/../lib/respond.php';

final class AdminOrgsController {
  public static function list(): void {
    require_role(['admin']);
    $rows = db()->query('SELECT id, name, short_name, color, is_active, contact_json, settings_json, created_at, updated_at
      FROM org ORDER BY is_active DESC, name ASC')->fetchAll();
    foreach ($rows as &$r) {
      $r['contact'] = json_decode($r['contact_json'] ?? '{}', true);
      $r['settings'] = json_decode($r['settings_json'] ?? '{}', true);
      unset($r['contact_json'], $r['settings_json']);
    }
    api_json(['ok'=>true, 'orgs'=>$rows]);
  }

  public static function create(): void {
    require_role(['admin']);
    $in = json_input();
    $name = trim((string)($in['name'] ?? ''));
    if ($name === '') api_error(400, 'name required');
    $short = trim((string)($in['short_name'] ?? ''));
    $color = trim((string)($in['color'] ?? ''));
    $contact = $in['contact'] ?? [];
    $settings = $in['settings'] ?? [];
    if (!is_array($contact)) $contact=[];
    if (!is_array($settings)) $settings=[];

    $stmt = db()->prepare('INSERT INTO org(id, name, short_name, color, is_active, contact_json, settings_json, created_at)
      VALUES (gen_random_uuid(), :n, NULLIF(:s,\'\'), NULLIF(:c,\'\'), true, :cj::jsonb, :sj::jsonb, now())
      RETURNING id');
    $stmt->execute([
      ':n'=>$name,
      ':s'=>$short,
      ':c'=>$color,
      ':cj'=>json_encode($contact, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
      ':sj'=>json_encode($settings, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES),
    ]);
    $id = (string)$stmt->fetchColumn();

    EventsController::emitGlobal('orgs.updated', ['org_id'=>$id]);
    api_json(['ok'=>true, 'id'=>$id], 201);
  }

  public static function update(string $orgId): void {
    require_role(['admin']);
    $in = json_input();

    $fields = [];
    $params = [':id'=>$orgId];
    if (array_key_exists('name', $in)) { $fields[]='name=:n'; $params[':n']=trim((string)$in['name']); }
    if (array_key_exists('short_name', $in)) { $fields[]='short_name=NULLIF(:s,\'\')'; $params[':s']=trim((string)$in['short_name']); }
    if (array_key_exists('color', $in)) { $fields[]='color=NULLIF(:c,\'\')'; $params[':c']=trim((string)$in['color']); }
    if (array_key_exists('is_active', $in)) { $fields[]='is_active=:a'; $params[':a']=(bool)$in['is_active']; }
    if (array_key_exists('contact', $in) && is_array($in['contact'])) { $fields[]='contact_json=:cj::jsonb'; $params[':cj']=json_encode($in['contact'], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); }
    if (array_key_exists('settings', $in) && is_array($in['settings'])) { $fields[]='settings_json=:sj::jsonb'; $params[':sj']=json_encode($in['settings'], JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); }

    if (!$fields) api_error(400, 'nothing to update');
    $sql = 'UPDATE org SET '.implode(',', $fields).', updated_at=now() WHERE id=:id';
    db()->prepare($sql)->execute($params);

    EventsController::emitGlobal('orgs.updated', ['org_id'=>$orgId]);
    api_json(['ok'=>true]);
  }

  public static function deactivate(string $orgId): void {
    require_role(['admin']);
    // Soft-deactivate only.
    db()->prepare('UPDATE org SET is_active=false, updated_at=now() WHERE id=:id')->execute([':id'=>$orgId]);
    EventsController::emitGlobal('orgs.updated', ['org_id'=>$orgId]);
    api_json(['ok'=>true]);
  }
}
